"""
Spectates an IRC channel.
"""
import socket
import logging
import sys
import time
import datetime
import re
import os

# configs
# these configs can be changed by config.json
ALLOWED_TYPES = (int, float, bool, str, dict, list, type(None))
server = ""
channel = ""
nickname = "ChanSpec"
LOG_LEVEL = 10
REJOIN_ON_KICK = True
CONNECTED = False
tries = 0
INITIAL_MESSAGES = ""
HEARTBEAT_INTERVAL = 10 # Setting this to 0 disables heartbeat, making the
                        # spectator hang when it's disconnected for timeout.
REQUEST_DELAY = 0.1 # Setting this to 0 enables blocking. This also
                    # disables heartbeat.
HEARTBEAT_TOLERANCE = 5 # If the server doesn't respond to this much PINGs,
                        # assume that it has disconnected.
MESSAGE_TIMEOUT = 28800 # The amount of message silence seconds before
                        # disconnection.
INCLUDE_PONGS = False # Don't include PONGs on the raw log.
configs = [k for k, v in globals().items() if type(v) in ALLOWED_TYPES and not k.startswith("_")]
# config stuff stop here

# things for logging
logger = logging.getLogger(nickname)
raw_logger = logging.getLogger(nickname + "-raw")
logging.root.setLevel(LOG_LEVEL)
DEFAULTS = lambda name="": {"user": "LOGGER", "command": name}
lastPing = datetime.datetime.now()


# make formatter
from streams import ColoredStream, FileStream
logger.addHandler(ColoredStream(LOG_LEVEL))
logger.addHandler(FileStream(LOG_LEVEL))
raw_logger.addHandler(FileStream(LOG_LEVEL, file="raw.log", format="[%(asctime)s] %(message)s"))

irc = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
irc.setblocking(0 if REQUEST_DELAY else 1)


# connect to the server
def connect():
    global CONNECTED
    irc.setblocking(1)
    logger.debug(f"Connecting to {server}", extra=DEFAULTS())
    irc.connect((server, 6667))
    raw_logger.info(f"CONNECT {server}")
    logger.debug(f"Connected to {server}", extra=DEFAULTS())
    irc.send(f"USER {nickname} {nickname} somewhere :Channel Spectator\n".encode())
    irc.send(f"NICK {nickname}\n".encode())
    irc.send(f"JOIN {channel}\n".encode())
    irc.send(INITIAL_MESSAGES.format(**os.environ, **globals()).encode())
    CONNECTED = True
    irc.setblocking(0 if REQUEST_DELAY else 1)


# disconnect to the server
def disconnect():
    global CONNECTED, irc
    try:
        irc.close()
    except Exception as e:
        logger.error(str(e), extras=DEFAULTS(type(e).__name))
    irc = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    CONNECTED = False


# log messages
def listen():
    global logger, tries, lastPing
    msg_str = lambda a: " ".join(a).lstrip(":")
    _nickname = nickname

    tries = 0
    while True:
        text = b""
        delay = 0
        pings = 0
        while not text.endswith(b"\r\n"):
            try:
                text += irc.recv(4096)
            except BlockingIOError as e:
                if "[Errno 11]" not in str(e): # Resource temporarily unavailable (buffer is empty)
                    raise e
            
            if REQUEST_DELAY:
                time.sleep(REQUEST_DELAY)
                delay += REQUEST_DELAY
                if delay > HEARTBEAT_INTERVAL and HEARTBEAT_INTERVAL != 0:
                    # dub dub
                    if pings >= HEARTBEAT_TOLERANCE:
                        raise TimeoutError(f"{server} doesn't respond to PINGs {HEARTBEAT_TOLERANCE} times")
                    try: # Send a Maybe Request
                        irc.send(f'PING :heartbeat\n'.encode())
                    except (BlockingIOError, BrokenPipeError) as e: 
                        if "broken pipe" not in str(e).lower():
                            raise e
                    pings += 1
                    delay = 0
                if delay > MESSAGE_TIMEOUT:
                    raise TimeoutError(f"{server} is silent for more than {MESSAGE_TIMEOUT} seconds")
        try:
            text = text.decode("utf8")
        except UnicodeDecodeError:
            text = text.decode("cp819") # use a fallback encoding system
        finally:
            text = text.split("\r\n") # split to individual commands

        for line in text:
            #print(line)
            if line == "": continue
            elif line.startswith('PING'):
                # Serve ball back to server
                #print(f'PONG {line.split()[1]}\n')
                irc.send(f'PONG {line.split()[1]}\n'.encode())
            elif line.startswith('PONG'):
                # Catch the ball instead of sending it
                lastPing = datetime.datetime.now()
                pings = 0
            elif line[0] == ":":
                # Log message
                match = re.match(r"^:(.+?)(?:!(.+?))?(?:@(.+?))? (.+)$", line)
                name, user, host, command = match.groups()
                command = command.split(" ")
                EXTRAS = {"user": name, "command": command[0]}
                if command[0] != "PONG" or INCLUDE_PONGS:
                    raw_logger.info(line)
                if command[0] == "JOIN":
                    # user joined
                    logger.info(f"Joined to {command[1]}", extra=EXTRAS)
                elif command[0] == "NOTICE":
                    # a notice
                    if command[1] in (nickname, "*"):
                        logger.warning(msg_str(command[2:]), extra=EXTRAS)
                elif command[0] == "PRIVMSG":
                    # a message
                    logger.info(msg_str(command[2:]), extra=EXTRAS)
                elif command[0] == "WALLOPS":
                    # a message to all ops
                    logger.warning(f"WALLOPS: {msg_str(command[2:])}", extra=EXTRAS)
                elif command[0] == "AWAY":
                    # user is away
                    reason = msg_str(command[1:])
                    logger.info(f"Marked as away{': %s'%reason if reason else ''}", extra=EXTRAS)
                elif command[0] == "BACK":
                    # user is back
                    logger.info("No longer away", extra=EXTRAS)
                elif command[0] == "MODE":
                    # sudo chmod
                    ch = command[1]
                    mode = command[2]
                    to = command[3] if len(command) == 4 else ""
                    logger.info(f"Set {mode} to {to if to else ch}", extra=EXTRAS)
                elif command[0] == "KICK":
                    # kicked
                    reason = msg_str(command[3:])
                    if command[2] == nickname:
                        # bot is kick
                        logger.error(f"Kicked bot!{' (Reason: %s)'%reason if reason else ''}", extra=EXTRAS)
                        if REJOIN_ON_KICK:
                            # rejoin to keep logging
                            logger.warning("Rejoining...", extra=EXTRAS)
                            irc.send(f"JOIN {channel}\n".encode())
                            # say that the bot gives no harm
                            irc.send(f"PRIVMSG {channel} :Please don't kick me: I'm an innocent spectator...\n".encode())
                        else: sys.exit(1)
                    else:
                        logger.warning(f"Kicked {command[2]}{' (Reason: %s)'%reason if reason else ''}", extra=EXTRAS)
                elif command[0] == "PART":
                    # bye bye channel
                    logger.info(f"Left {channel}", extra=EXTRAS)
                elif command[0] == "QUIT":
                    # bye bye server
                    reason = msg_str(command[1:])
                    logger.info(f"Client quit{': %s'%reason if reason else ''}", extra=EXTRAS)
                    if user == nickname:
                        raise ConnectionAbortedError(reason)
                elif command[0] == "PONG":
                    # Catch the ball instead of sending it
                    lastPing = datetime.datetime.now()
                    pings = 0
                elif re.match(r"\d{3}", command[0]):
                    # cryptic 3-number code
                    code = int(command[0])
                    if code > 400:
                        # uh oh
                        if code == 465: # ERR_YOUREBANNEDCREEP
                            raise ConnectionRefusedError("You are banned from this server!")
                        elif code == 433: # ERR_NICKNAMEINUSE
                            logger.warning(f"Nickname {_nickname} is already in use! "
                                           f"Trying {_nickname + '_'} instead...", extra=EXTRAS)
                            _nickname += "_"
                            irc.send(f"NICK {_nickname}\n".encode())
                            # most likely the bot isn't on the channel so rejoin
                            irc.send(f"JOIN {channel}\n".encode())
                        else:
                            logger.error(" ".join(command), extra=EXTRAS)
                    else:
                        logger.debug(" ".join(command), extra=EXTRAS)
                else:
                    # *shrug*
                    logger.debug(line, extra=EXTRAS)


def setConfig(glob=globals()):
    global configs
    import json
    with open("config.json", "r") as f:
        config = json.loads(f.read())
        config = {k: v for k, v in config.items() if k in configs}
        glob.update(config)


def main():
    global tries, CONNECTED

    setConfig()
    def reconnect():
        try: disconnect()
        except: pass
        try: connect()
        except: pass

    try:
        lastError = None
        while True:
            try:
                if not CONNECTED: connect()
                last_error = None
                listen()
            except ConnectionRefusedError:
                raise
            except (OSError, socket.gaierror) as e:
                if CONNECTED and lastError != e:
                    raw_logger.info(f"DISCONNECT :{e}")
                    last_error = e
                # connection problems?
                tries += 1
                if tries < 10:
                    logger.error(f"Failed at receiving data: {e}. Retrying (try {tries})", extra=DEFAULTS("ERROR"))
                elif tries == 10:
                    logger.error(f"Failed at receiving data: {e}. Retrying (try {tries-1}+)", extra=DEFAULTS("ERROR"))
                # resolve, resolve.
                if "not connect" in str(e):
                    # something about not being connected
                    CONNECTED = False
                    reconnect()
                    time.sleep(4)
                if "reset by peer" in str(e):
                    # something about connection reset by peer
                    CONNECTED = False
                    reconnect()
                    time.sleep(4)
                elif "already connect" in str(e):
                    # something about already connected
                    CONNECTED = True
                    continue
                elif "abort" in str(e):
                    # something about connection aborted
                    CONNECTED = False
                    reconnect()
                    time.sleep(4)
                elif "timed out" in str(e) or type(e) is TimeoutError:
                    # something about connection timed out
                    logger.warning(f"{channel} hasn't been logged since {lastPing}!", extra=DEFAULTS("WARNING"))
                    CONNECTED = False
                    reconnect()
                    time.sleep(4)
                elif not CONNECTED:
                    # connection problems
                    time.sleep(4)
                time.sleep(1)
                continue
            except SystemExit:
                # okthxbye
                logger.critical(f"Received SystemExit, exiting.", extra=DEFAULTS("CRITICAL"))
                raise
    except Exception as e:
        # uh oh
        logger.critical(f"Logger stopped because of an error: {type(e).__name__}: {e}", extra=DEFAULTS("CRITICAL"))
        if CONNECTED: irc.send(f"QUIT :{type(e).__name__}: {e}\n".encode())
        raise e

if __name__ == "__main__":
    main()
